/*
 * Decompiled with CFR 0.152.
 */
package ghidra.feature.fid.db;

import db.DBHandle;
import db.DBRecord;
import ghidra.feature.fid.db.FidFile;
import ghidra.feature.fid.db.FunctionRecord;
import ghidra.feature.fid.db.FunctionsTable;
import ghidra.feature.fid.db.LibrariesTable;
import ghidra.feature.fid.db.LibraryRecord;
import ghidra.feature.fid.db.RelationType;
import ghidra.feature.fid.db.RelationsTable;
import ghidra.feature.fid.db.StringsTable;
import ghidra.feature.fid.hash.FidHashQuad;
import ghidra.framework.store.db.PackedDBHandle;
import ghidra.framework.store.db.PackedDatabase;
import ghidra.program.model.lang.CompilerSpecID;
import ghidra.program.model.lang.LanguageID;
import ghidra.util.Msg;
import ghidra.util.ReadOnlyException;
import ghidra.util.exception.AssertException;
import ghidra.util.exception.CancelledException;
import ghidra.util.exception.DuplicateFileException;
import ghidra.util.exception.VersionException;
import ghidra.util.task.TaskMonitor;
import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.atomic.AtomicInteger;

public class FidDB
implements Closeable {
    private static final String FID_CONTENT_TYPE = "Function ID Database";
    private static final String[] OLD_V1_FID_TABLES = new String[]{"Relations Table", "Functions Table", "Strings Table", "Libraries Table"};
    private final FidFile fidFile;
    private final DBHandle handle;
    private LibrariesTable librariesTable;
    private StringsTable stringsTable;
    private FunctionsTable functionsTable;
    private RelationsTable relationsTable;
    private long openTransaction;
    private boolean openForUpdate;
    private AtomicInteger openCount = new AtomicInteger();

    FidDB(FidFile fidFile, boolean openForUpdate) throws IOException, VersionException {
        this.fidFile = fidFile;
        this.openForUpdate = openForUpdate;
        this.openCount.set(1);
        this.handle = fidFile.isInstalled() ? this.openRawDatabaseFile() : this.openPackedDatabase();
        this.getTables();
        if (openForUpdate) {
            this.openTransaction = this.handle.startTransaction();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    static void createNewFidDatabase(File file) throws IOException {
        if (file.exists()) {
            throw new DuplicateFileException("File already exists: " + file.getAbsolutePath());
        }
        try (PackedDBHandle packedDBHandle = new PackedDBHandle(FID_CONTENT_TYPE);){
            long txId = packedDBHandle.startTransaction();
            LibrariesTable.createTable((DBHandle)packedDBHandle);
            StringsTable.createTable((DBHandle)packedDBHandle);
            FunctionsTable.createTable((DBHandle)packedDBHandle);
            RelationsTable.createTables((DBHandle)packedDBHandle);
            packedDBHandle.endTransaction(txId, true);
            String name = file.getName();
            packedDBHandle.saveAs(name, file.getParentFile(), name, TaskMonitor.DUMMY);
        }
    }

    public void saveRawDatabaseFile(File file, TaskMonitor monitor) throws CancelledException, IOException {
        this.handle.saveAs(file, false, monitor);
    }

    private DBHandle openRawDatabaseFile() throws IOException {
        this.openForUpdate = false;
        return new DBHandle(this.fidFile.getFile());
    }

    private DBHandle openPackedDatabase() throws IOException {
        try {
            PackedDatabase pdb = PackedDatabase.getPackedDatabase((File)this.fidFile.getFile(), (boolean)false, (TaskMonitor)TaskMonitor.DUMMY);
            if (this.openForUpdate) {
                return pdb.openForUpdate(TaskMonitor.DUMMY);
            }
            return pdb.open(TaskMonitor.DUMMY);
        }
        catch (CancelledException cancelledException) {
            throw new AssertException("Can't happen!");
        }
    }

    private void getTables() throws IOException, VersionException {
        this.librariesTable = new LibrariesTable(this.handle);
        this.stringsTable = new StringsTable(this.handle);
        this.functionsTable = new FunctionsTable(this, this.handle);
        this.relationsTable = new RelationsTable(this.handle);
    }

    public String getName() {
        return this.fidFile.getName();
    }

    public String getPath() {
        return this.fidFile.getPath();
    }

    public String toString() {
        return "FidDB: " + this.fidFile.getFile().getAbsolutePath();
    }

    public void incrementOpenCount() {
        this.openCount.incrementAndGet();
    }

    @Override
    public void close() {
        if (this.openCount.decrementAndGet() == 0) {
            this.fidFile.closingFidDB(this);
            try {
                if (this.openForUpdate) {
                    this.handle.endTransaction(this.openTransaction, true);
                }
                this.handle.close();
            }
            catch (IOException e) {
                Msg.error((Object)this, (Object)("Error closing " + String.valueOf(this)), (Throwable)e);
            }
            this.librariesTable = null;
            this.stringsTable = null;
            this.functionsTable = null;
            this.relationsTable = null;
        }
    }

    StringsTable getStringsTable() {
        return this.stringsTable;
    }

    public List<LibraryRecord> getAllLibraries() {
        List<Object> libraries;
        try {
            libraries = this.librariesTable == null ? Collections.emptyList() : this.librariesTable.getLibraries();
        }
        catch (IOException e) {
            Msg.error((Object)this, (Object)"Error in FID database", (Throwable)e);
            libraries = Collections.emptyList();
        }
        return libraries;
    }

    public List<FunctionRecord> findFunctionsByLibraryAndName(LibraryRecord library, String name) {
        try {
            List<FunctionRecord> list = this.functionsTable.getFunctionRecordsByLibraryAndName(library, name);
            return list;
        }
        catch (IOException e) {
            Msg.error((Object)this, (Object)"Serious problem searching for FID Functions by name and namespace", (Throwable)e);
            return null;
        }
    }

    public List<FunctionRecord> findFunctionsByNameSubstring(String name) {
        try {
            List<FunctionRecord> list = this.functionsTable.getFunctionRecordsByNameSubstring(name);
            return list;
        }
        catch (IOException e) {
            Msg.error((Object)this, (Object)"Serious problem searching for FID Functions by name substring", (Throwable)e);
            return null;
        }
    }

    public List<FunctionRecord> findFunctionsByNameRegex(String regex) {
        try {
            List<FunctionRecord> list = this.functionsTable.getFunctionRecordsByNameRegex(regex);
            return list;
        }
        catch (IOException e) {
            Msg.error((Object)this, (Object)"Serious problem search for FID Functions by regular expression", (Throwable)e);
            return null;
        }
    }

    public List<FunctionRecord> findFunctionsByDomainPathSubstring(String domainPath) {
        try {
            List<FunctionRecord> list = this.functionsTable.getFunctionRecordsByDomainPathSubstring(domainPath);
            return list;
        }
        catch (IOException e) {
            Msg.error((Object)this, (Object)"Serious problem searching for FID Functions by domain path", (Throwable)e);
            return null;
        }
    }

    public Long findFullHashValueAtOrAfter(long value) {
        try {
            Long search = this.functionsTable.getFullHashValueAtOrAfter(value);
            return search;
        }
        catch (IOException e) {
            Msg.error((Object)this, (Object)"Serious problem searching for full hash values", (Throwable)e);
            return null;
        }
    }

    public List<FunctionRecord> findFunctionsBySpecificHash(long specificHash) {
        try {
            List<FunctionRecord> list = this.functionsTable.getFunctionRecordsBySpecificHash(specificHash);
            return list;
        }
        catch (IOException e) {
            Msg.error((Object)this, (Object)"Serious problem searching for FID Functions by specific hash", (Throwable)e);
            return null;
        }
    }

    public List<FunctionRecord> findFunctionsByFullHash(long fullHash) {
        try {
            List<FunctionRecord> list = this.functionsTable.getFunctionRecordsByFullHash(fullHash);
            return list;
        }
        catch (IOException e) {
            Msg.error((Object)this, (Object)"Serious problem searching for FID Functions by full hash", (Throwable)e);
            return null;
        }
    }

    public List<LibraryRecord> findLibrariesByName(String family, String version, String variant) {
        try {
            List<LibraryRecord> list = this.librariesTable.getLibrariesByName(family, version, variant);
            return list;
        }
        catch (IOException e) {
            Msg.error((Object)this, (Object)"Serious problem search for FID Libraries by name", (Throwable)e);
            return null;
        }
    }

    public boolean getSuperiorFullRelation(FunctionRecord superiorFunction, FidHashQuad inferiorFunction) {
        try {
            DBRecord libraryByID = this.librariesTable.getLibraryByID(superiorFunction.getLibraryID());
            if (libraryByID != null) {
                return this.relationsTable.getSuperiorFullRelation(superiorFunction, inferiorFunction);
            }
        }
        catch (IOException e) {
            Msg.error((Object)this, (Object)"Serious problem in getSuperiorFullRelation", (Throwable)e);
        }
        return false;
    }

    public boolean getInferiorFullRelation(FidHashQuad superiorFunction, FunctionRecord inferiorFunction) {
        try {
            DBRecord libraryByID = this.librariesTable.getLibraryByID(inferiorFunction.getLibraryID());
            if (libraryByID != null) {
                return this.relationsTable.getInferiorFullRelation(superiorFunction, inferiorFunction);
            }
        }
        catch (IOException e) {
            Msg.error((Object)this, (Object)"Serious problem in getInferiorFullRelation", (Throwable)e);
        }
        return false;
    }

    public FunctionRecord getFunctionByID(long functionID) {
        try {
            FunctionRecord functionRecord = this.functionsTable.getFunctionByID(functionID);
            return functionRecord;
        }
        catch (IOException e) {
            Msg.error((Object)this, (Object)"Serious problem finding Function record by ID", (Throwable)e);
            return null;
        }
    }

    public LibraryRecord getLibraryForFunction(FunctionRecord functionRecord) {
        try {
            DBRecord record = this.librariesTable.getLibraryByID(functionRecord.getLibraryID());
            if (record == null) {
                return null;
            }
            return new LibraryRecord(record);
        }
        catch (IOException e) {
            Msg.error((Object)this, (Object)"Serious problem finding Library for function", (Throwable)e);
            return null;
        }
    }

    public DBHandle getDBHandle() {
        return this.handle;
    }

    public LibraryRecord createNewLibrary(String libraryFamilyName, String libraryVersion, String libraryVariant, String ghidraVersion, LanguageID languageID, int languageVersion, int languageMinorVersion, CompilerSpecID compilerSpecID) {
        try {
            this.checkUpdateAllowed();
            DBRecord record = this.librariesTable.createLibrary(libraryFamilyName, libraryVersion, libraryVariant, ghidraVersion, languageID, languageVersion, languageMinorVersion, compilerSpecID);
            return new LibraryRecord(record);
        }
        catch (ReadOnlyException e) {
            Msg.error((Object)this, (Object)((Object)e));
        }
        catch (IOException e) {
            Msg.error((Object)this, (Object)"Serious problem creating FID Library record", (Throwable)e);
        }
        return null;
    }

    private void checkUpdateAllowed() throws ReadOnlyException {
        if (!this.openForUpdate) {
            throw new ReadOnlyException("Attempted to modify Fid Database that is not open for update: " + String.valueOf(this));
        }
    }

    public FunctionRecord createNewFunction(LibraryRecord library, FidHashQuad hashQuad, String name, long entryPoint, String domainPath, boolean hasTerminator) {
        try {
            this.checkUpdateAllowed();
            FunctionRecord functionRecord = this.functionsTable.createFunctionRecord(library.getLibraryID(), hashQuad, name, entryPoint, domainPath, hasTerminator);
            return functionRecord;
        }
        catch (ReadOnlyException e) {
            Msg.error((Object)this, (Object)((Object)e));
        }
        catch (IOException e) {
            Msg.error((Object)this, (Object)"Serious problem creating FID Function record", (Throwable)e);
        }
        return null;
    }

    public void createRelation(FunctionRecord superiorFunction, FunctionRecord inferiorFunction, RelationType relationType) {
        try {
            this.checkUpdateAllowed();
            this.relationsTable.createRelation(superiorFunction, inferiorFunction, relationType);
        }
        catch (ReadOnlyException e) {
            Msg.error((Object)this, (Object)((Object)e));
        }
        catch (IOException e) {
            Msg.error((Object)this, (Object)"Serious problem creating FID Relation record", (Throwable)e);
        }
    }

    public void createInferiorRelation(FunctionRecord superiorFunction, FunctionRecord inferiorFunction) {
        try {
            this.checkUpdateAllowed();
            this.relationsTable.createInferiorRelation(superiorFunction, inferiorFunction);
        }
        catch (ReadOnlyException e) {
            Msg.error((Object)this, (Object)((Object)e));
        }
        catch (IOException e) {
            Msg.error((Object)this, (Object)"Serious problem creating FID Inferior Relation record", (Throwable)e);
        }
    }

    private void modifyFlags(List<FunctionRecord> funcList, int flagMask, boolean value) throws IOException {
        for (FunctionRecord funcRec : funcList) {
            this.functionsTable.modifyFlags(funcRec.getID(), flagMask, value);
        }
    }

    private FunctionRecord modifyFunctionFlag(FunctionRecord funcRec, int flagMask, boolean value) throws IOException {
        if (funcRec.getFidDb() != this) {
            throw new IOException("Mismatched FunctionRecord and FidDb");
        }
        long key = funcRec.getID();
        this.functionsTable.modifyFlags(key, flagMask, value);
        FunctionRecord res = this.functionsTable.getFunctionByID(key);
        if (res == null) {
            throw new IOException("Could not recover modified FunctionRecord");
        }
        return res;
    }

    public void setAutoPassByFullHash(long hash, boolean value) throws IOException {
        this.checkUpdateAllowed();
        List<FunctionRecord> funcList = this.findFunctionsByFullHash(hash);
        this.modifyFlags(funcList, 2, value);
    }

    public void setAutoFailByFullHash(long hash, boolean value) throws IOException {
        this.checkUpdateAllowed();
        List<FunctionRecord> funcList = this.findFunctionsByFullHash(hash);
        this.modifyFlags(funcList, 4, value);
    }

    public void setForceSpecificByFullHash(long hash, boolean value) throws IOException {
        this.checkUpdateAllowed();
        List<FunctionRecord> funcList = this.findFunctionsByFullHash(hash);
        this.modifyFlags(funcList, 8, value);
    }

    public void setForceRelationByFullHash(long hash, boolean value) throws IOException {
        this.checkUpdateAllowed();
        List<FunctionRecord> funcList = this.findFunctionsByFullHash(hash);
        this.modifyFlags(funcList, 16, value);
    }

    public FunctionRecord setAutoPassOnFunction(FunctionRecord funcRec, boolean value) throws IOException {
        this.checkUpdateAllowed();
        return this.modifyFunctionFlag(funcRec, 2, value);
    }

    public FunctionRecord setAutoFailOnFunction(FunctionRecord funcRec, boolean value) throws IOException {
        this.checkUpdateAllowed();
        return this.modifyFunctionFlag(funcRec, 4, value);
    }

    public FunctionRecord setForceSpecificOnFunction(FunctionRecord funcRec, boolean value) throws IOException {
        this.checkUpdateAllowed();
        return this.modifyFunctionFlag(funcRec, 8, value);
    }

    public FunctionRecord setForceRelationOnFunction(FunctionRecord funcRec, boolean value) throws IOException {
        this.checkUpdateAllowed();
        return this.modifyFunctionFlag(funcRec, 16, value);
    }

    public void setAutoPassByName(String library, String version, String variant, String functionName, boolean value) throws IOException {
        this.checkUpdateAllowed();
        List<LibraryRecord> libraryList = this.findLibrariesByName(library, version, variant);
        for (LibraryRecord libRec : libraryList) {
            List<FunctionRecord> funcList = this.findFunctionsByLibraryAndName(libRec, functionName);
            this.modifyFlags(funcList, 2, value);
        }
    }

    public void setAutoFailByName(String library, String version, String variant, String functionName, boolean value) throws IOException {
        this.checkUpdateAllowed();
        List<LibraryRecord> libraryList = this.findLibrariesByName(library, version, variant);
        for (LibraryRecord libRec : libraryList) {
            List<FunctionRecord> funcList = this.findFunctionsByLibraryAndName(libRec, functionName);
            this.modifyFlags(funcList, 4, value);
        }
    }

    public void setForceSpecificByName(String library, String version, String variant, String functionName, boolean value) throws IOException {
        this.checkUpdateAllowed();
        List<LibraryRecord> libraryList = this.findLibrariesByName(library, version, variant);
        for (LibraryRecord libRec : libraryList) {
            List<FunctionRecord> funcList = this.findFunctionsByLibraryAndName(libRec, functionName);
            this.modifyFlags(funcList, 8, value);
        }
    }

    public void setForceRelationByName(String library, String version, String variant, String functionName, boolean value) throws IOException {
        this.checkUpdateAllowed();
        List<LibraryRecord> libraryList = this.findLibrariesByName(library, version, variant);
        for (LibraryRecord libRec : libraryList) {
            List<FunctionRecord> funcList = this.findFunctionsByLibraryAndName(libRec, functionName);
            this.modifyFlags(funcList, 16, value);
        }
    }

    public void saveDatabase(String comment, TaskMonitor monitor) throws IOException, CancelledException {
        if (!this.openForUpdate) {
            return;
        }
        this.handle.endTransaction(this.openTransaction, true);
        this.handle.save(comment, null, monitor);
        this.openTransaction = this.handle.startTransaction();
    }
}

