﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/ec2/EC2Request.h>
#include <aws/ec2/EC2_EXPORTS.h>
#include <aws/ec2/model/TagSpecification.h>

#include <utility>

namespace Aws {
namespace EC2 {
namespace Model {

/**
 */
class CreateTransitGatewayRouteTableAnnouncementRequest : public EC2Request {
 public:
  AWS_EC2_API CreateTransitGatewayRouteTableAnnouncementRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateTransitGatewayRouteTableAnnouncement"; }

  AWS_EC2_API Aws::String SerializePayload() const override;

 protected:
  AWS_EC2_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>The ID of the transit gateway route table.</p>
   */
  inline const Aws::String& GetTransitGatewayRouteTableId() const { return m_transitGatewayRouteTableId; }
  inline bool TransitGatewayRouteTableIdHasBeenSet() const { return m_transitGatewayRouteTableIdHasBeenSet; }
  template <typename TransitGatewayRouteTableIdT = Aws::String>
  void SetTransitGatewayRouteTableId(TransitGatewayRouteTableIdT&& value) {
    m_transitGatewayRouteTableIdHasBeenSet = true;
    m_transitGatewayRouteTableId = std::forward<TransitGatewayRouteTableIdT>(value);
  }
  template <typename TransitGatewayRouteTableIdT = Aws::String>
  CreateTransitGatewayRouteTableAnnouncementRequest& WithTransitGatewayRouteTableId(TransitGatewayRouteTableIdT&& value) {
    SetTransitGatewayRouteTableId(std::forward<TransitGatewayRouteTableIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the peering attachment.</p>
   */
  inline const Aws::String& GetPeeringAttachmentId() const { return m_peeringAttachmentId; }
  inline bool PeeringAttachmentIdHasBeenSet() const { return m_peeringAttachmentIdHasBeenSet; }
  template <typename PeeringAttachmentIdT = Aws::String>
  void SetPeeringAttachmentId(PeeringAttachmentIdT&& value) {
    m_peeringAttachmentIdHasBeenSet = true;
    m_peeringAttachmentId = std::forward<PeeringAttachmentIdT>(value);
  }
  template <typename PeeringAttachmentIdT = Aws::String>
  CreateTransitGatewayRouteTableAnnouncementRequest& WithPeeringAttachmentId(PeeringAttachmentIdT&& value) {
    SetPeeringAttachmentId(std::forward<PeeringAttachmentIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The tags specifications applied to the transit gateway route table
   * announcement.</p>
   */
  inline const Aws::Vector<TagSpecification>& GetTagSpecifications() const { return m_tagSpecifications; }
  inline bool TagSpecificationsHasBeenSet() const { return m_tagSpecificationsHasBeenSet; }
  template <typename TagSpecificationsT = Aws::Vector<TagSpecification>>
  void SetTagSpecifications(TagSpecificationsT&& value) {
    m_tagSpecificationsHasBeenSet = true;
    m_tagSpecifications = std::forward<TagSpecificationsT>(value);
  }
  template <typename TagSpecificationsT = Aws::Vector<TagSpecification>>
  CreateTransitGatewayRouteTableAnnouncementRequest& WithTagSpecifications(TagSpecificationsT&& value) {
    SetTagSpecifications(std::forward<TagSpecificationsT>(value));
    return *this;
  }
  template <typename TagSpecificationsT = TagSpecification>
  CreateTransitGatewayRouteTableAnnouncementRequest& AddTagSpecifications(TagSpecificationsT&& value) {
    m_tagSpecificationsHasBeenSet = true;
    m_tagSpecifications.emplace_back(std::forward<TagSpecificationsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Checks whether you have the required permissions for the action, without
   * actually making the request, and provides an error response. If you have the
   * required permissions, the error response is <code>DryRunOperation</code>.
   * Otherwise, it is <code>UnauthorizedOperation</code>.</p>
   */
  inline bool GetDryRun() const { return m_dryRun; }
  inline bool DryRunHasBeenSet() const { return m_dryRunHasBeenSet; }
  inline void SetDryRun(bool value) {
    m_dryRunHasBeenSet = true;
    m_dryRun = value;
  }
  inline CreateTransitGatewayRouteTableAnnouncementRequest& WithDryRun(bool value) {
    SetDryRun(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_transitGatewayRouteTableId;

  Aws::String m_peeringAttachmentId;

  Aws::Vector<TagSpecification> m_tagSpecifications;

  bool m_dryRun{false};
  bool m_transitGatewayRouteTableIdHasBeenSet = false;
  bool m_peeringAttachmentIdHasBeenSet = false;
  bool m_tagSpecificationsHasBeenSet = false;
  bool m_dryRunHasBeenSet = false;
};

}  // namespace Model
}  // namespace EC2
}  // namespace Aws
