﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/states/SFNRequest.h>
#include <aws/states/SFN_EXPORTS.h>

#include <utility>

namespace Aws {
namespace SFN {
namespace Model {

/**
 */
class PublishStateMachineVersionRequest : public SFNRequest {
 public:
  AWS_SFN_API PublishStateMachineVersionRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "PublishStateMachineVersion"; }

  AWS_SFN_API Aws::String SerializePayload() const override;

  AWS_SFN_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the state machine.</p>
   */
  inline const Aws::String& GetStateMachineArn() const { return m_stateMachineArn; }
  inline bool StateMachineArnHasBeenSet() const { return m_stateMachineArnHasBeenSet; }
  template <typename StateMachineArnT = Aws::String>
  void SetStateMachineArn(StateMachineArnT&& value) {
    m_stateMachineArnHasBeenSet = true;
    m_stateMachineArn = std::forward<StateMachineArnT>(value);
  }
  template <typename StateMachineArnT = Aws::String>
  PublishStateMachineVersionRequest& WithStateMachineArn(StateMachineArnT&& value) {
    SetStateMachineArn(std::forward<StateMachineArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Only publish the state machine version if the current state machine's
   * revision ID matches the specified ID.</p> <p>Use this option to avoid publishing
   * a version if the state machine changed since you last updated it. If the
   * specified revision ID doesn't match the state machine's current revision ID, the
   * API returns <code>ConflictException</code>.</p>  <p>To specify an initial
   * revision ID for a state machine with no revision ID assigned, specify the string
   * <code>INITIAL</code> for the <code>revisionId</code> parameter. For example, you
   * can specify a <code>revisionID</code> of <code>INITIAL</code> when you create a
   * state machine using the <a>CreateStateMachine</a> API action.</p>
   */
  inline const Aws::String& GetRevisionId() const { return m_revisionId; }
  inline bool RevisionIdHasBeenSet() const { return m_revisionIdHasBeenSet; }
  template <typename RevisionIdT = Aws::String>
  void SetRevisionId(RevisionIdT&& value) {
    m_revisionIdHasBeenSet = true;
    m_revisionId = std::forward<RevisionIdT>(value);
  }
  template <typename RevisionIdT = Aws::String>
  PublishStateMachineVersionRequest& WithRevisionId(RevisionIdT&& value) {
    SetRevisionId(std::forward<RevisionIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An optional description of the state machine version.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  PublishStateMachineVersionRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_stateMachineArn;
  bool m_stateMachineArnHasBeenSet = false;

  Aws::String m_revisionId;
  bool m_revisionIdHasBeenSet = false;

  Aws::String m_description;
  bool m_descriptionHasBeenSet = false;
};

}  // namespace Model
}  // namespace SFN
}  // namespace Aws
